/********************************************************************
 * FileName:		FTP.c
 * Dependencies:
 * Processor:				
 * Hardware:		
 * Assembler:		
 * Linker:		
 * Company:		Microchip Technology, Inc.
 *
 * Software License Agreement:
 * The software supplied herewith by Microchip Technology Incorporated
 * (the CompanyE for its PICmicro Microcontroller is intended and
 * supplied to you, the Companys customer, for use solely and
 * exclusively on Microchip PICmicro Microcontroller products. The
 * software is owned by the Company and/or its supplier, and is
 * protected under applicable copyright laws. All rights are reserved.
 * Any use in violation of the foregoing restrictions may subject the
 * user to criminal sanctions under applicable laws, as well as to
 * civil liability for the breach of the terms and conditions of this
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN AS ISECONDITION. NO WARRANTIES,
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 *********************************************************************
 * File Description:
 *
 * FTP Server
 *
 * Change History:
 * Name				Date            Changes
 * Sean Justice		03/10/06		Initial Version
 ********************************************************************/

#include "TCPIP Stack/TCPIP.h"
#include "TCPIP Stack/StackTsk.h"
//#include "TCPIP Stack/ftp.h"
#include "FreeRTOS.h"
#include "task.h"
#include "MDD File System/FSIO.h"
#include "MDD File System/FSDefs.h"
#include <time.h>
#include <string.h>

/****************************************************************
 * private functions
 ****************************************************************/
static BOOL _AcceptConnection(void);
static void _CloseConnection(TCP_SOCKET socket);
static int _GetTCPRecv(TCP_SOCKET socket, BYTE *data, BYTE *data_len);
static BYTE _ParseFTPString(BYTE *ftp_str, BYTE **arg, BYTE max_arg);
static FTP_COMMAND _GetFTPCommand(BYTE *cmd_str);
static BOOL _ExecuteFTPCommand(FTP_COMMAND cmd, BYTE **arg);
static BOOL _FTPQuit(void);
static BOOL _FTPPutFile(BYTE *filename);
static int _SendData(TCP_SOCKET socket, BYTE *buf, int len);
static BOOL _FTPGetFile(BYTE *filename);
static BOOL _FTPDeleFile(BYTE *filename);
static BOOL _FTPListFile(BYTE *arg[]);
static	BOOL _FTPCWD(BYTE *directory);
static BOOL _FTPDeleDirectory(BYTE *directory);
static BOOL _FTPMakeDirectory(BYTE *directory);
static	BOOL _FTPPASV(void);

BOOL FTPVerify(BYTE *user, BYTE *password);
void	KeepAliveSend (void);

static	DWORD	RemoteIP;

static	BYTE	buf[64];
static	BYTE	size;
extern	UDP_SOCKET	DPlusSocket;
extern	DWORD	ReConnectTimer;
extern	BYTE	Sync[3];
extern	unsigned short int recordFoundName[257];
#define	PassivePort	(4095u)			// (LOCAL_UDP_PORT_END_NUMBER+1)

/************************************************************************
 * private constants
 ************************************************************************/
const char *_ftpCommandString[] =
{
    "USER",                         // FTP_CMD_USER
    "PASS",                         // FTP_CMD_PASS
    "QUIT",                         // FTP_CMD_QUIT
    "STOR",                         // FTP_CMD_STOR
    "PORT",                         // FTP_CMD_PORT
    "ABOR",                         // FTP_CMD_ABORT
	"SYST",							// FTP_CMD_SYST
	"NOOP",							// FTP_CMD_NOOP
	"PWD",							// FTP_CMD_PWD
	"TYPE",							// FTP_CMD_TYPE
	"RETR",							// FTP_CMD_RETR
	"DELE",							// FTP_CMD_DELE
	"NLST",							// FTP_CMD_NLST
	"LIST",							// FTP_CMD_LIST
	"CWD",							// FTP_CMD_CWD
	"XRMD",							// FTP_CMD_RMD
	"XMKD",							// FTP_CMD_MKD
	"PASV"							// FTP_CMD_PASV
};
#define FTP_COMMAND_TABLE_SIZE  ( sizeof(FTPCommandString)/sizeof(FTPCommandString[0]) )

const char *_ftpResponseString[] =
{
    "220 Ready\r\n",							// FTP_RESP_BANNER
    "331 Password required\r\n",				// FTP_RESP_USER_OK
    "230 Logged in\r\n",						// FTP_RESP_PASS_OK
    "221 Bye\r\n",								// FTP_RESP_QUIT_OK
    "500 \r\n",									// FTP_RESP_STOR_OK
    "502 Not implemented\r\n",					// FTP_RESP_UNKNOWN
    "530 Login required\r\n",					// FTP_RESP_LOGIN
    "150 Transferring data...\r\n",				// FTP_RESP_DATA_OPEN
    "125 Done.\r\n",							// FTP_RESP_DATA_READY
    "226 Transfer Complete\r\n",				// FTP_RESP_DATA_CLOSE
    "200 ok\r\n",								// FTP_RESP_OK
	"215 UNIX Type: I\r\n",						// FTP_RESP_SYST
	"200 Switching to ASCII mode.\r\n",			// ftp_RESP_ASCII
	"257 \"/\" is current directory\r\n",		// FTP_RESP_DIRECTORY
	"250 Delete operation successuful.\r\n",		// FTP_RESP_DELETE
	"150 Here comes the directory listing.\r\n",		// FTP_RESP_LIST
	"226 Directory send OK.\r\n",					// FTP_RESP_LIST_DONE
	"250 Directory successfully changed.\r\n",		// FTP_RESP_CWD
	"227 Entering Passive Mode ",					// FTP_RESP_PASV
	"200 PORT command successful.\r\n"				// FTP_RESP_PORT
};

/****************************************************************
 * private variables
 ****************************************************************/
static	TCP_SOCKET		_ftpListenSocket;
static	TCP_SOCKET		_ftpDataSocket;
static	TCP_SOCKET		_ftpConnectionSocket;
static	WORD_VAL		_ftpDataPort;
static	SM_FTP			_ftpStage;
static	SM_FTP_CMD		_ftpStageCmd;
static	FTP_COMMAND		_ftpCmd;
static	FTP_RESPONSE	_ftpResponce;	
static	BYTE			_ftpUser[FTP_USER_NAME_LEN];
static	BYTE			_ftpStr[MAX_FTP_CMD_STRING_LEN + 2];
static	BYTE			_ftpStrLen;
static	BYTE			*_ftpArgv[MAX_FTP_ARGS];
static	BYTE			_ftpNumArg;
static	DWORD			_lastActivity;
static	FSFILE			*_ftpFile;
static	FTP_FLAGS		_ftpFlags;
static	BOOL			_ftpConnection;
static	BYTE			_ftpTxBuf[FTP_TX_BUF_SIZE];
static	int				_ftpTxSize;
static	BOOL			_ftpPassive;

/*********************************************************************
 * Function:		PUBLIC BOOL FTPInit(void)       
 *
 * PreCondition:	The TCP/IP is initialized.  
 *
 * Input:           None.
 *
 * Output:          true if socket was successfully creasted. else false
 *
 * Side Effects:    None
 *
 * Overview:        This function will create a socket and bind it to
 *					the FTP port.  It will listen to incoming connection
 *					requests.
 ********************************************************************/
BOOL FTPInit(void)
{
    _ftpListenSocket = TCPOpen(0, TCP_OPEN_SERVER, FTP_COMMAND_PORT, TCP_PURPOSE_FTP_COMMAND);

    if(_ftpListenSocket == INVALID_SOCKET)
        return FALSE;

	_ftpStage				= SM_FTP_NOT_CONNECTED;
	_ftpStrLen				= 0;
	_ftpFlags.Val			= 0;
	_ftpDataPort.Val		= FTP_DATA_PORT;
	_ftpConnection			= FALSE;
	_ftpConnectionSocket	= INVALID_SOCKET;
	_ftpDataSocket			= INVALID_SOCKET;
	_ftpCmd					= FTP_CMD_NONE;
	_ftpStageCmd			= SM_FTP_CMD_IDLE;
	_ftpTxSize				= 0;
	_ftpFile				= NULL;

	_ftpPassive				= FALSE;

	return TRUE;
	
}
/*********************************************************************
 * Function:        PUBLIC BOOL FTPServer(void)
 *
 * PreCondition:    FTPInit has been called
 *
 * Input:           None.
 *
 * Output:          true if FTP connections are being serviced, else false
 *
 * Side Effects:    None.
 *
 * Overview:        This function handles incoming FTP connections and 
 *					services all incoming commands
 ********************************************************************/
void	FTPServer(void)
{
	int result;
	static	SOCKET_INFO	*remoteInfo;

	if(_ftpListenSocket == INVALID_SOCKET)
	{
		if (!FTPInit()) return;
	}

	// check to see if we have a connection pending
	if(!_ftpConnection)
	{
		_ftpConnection = _AcceptConnection();
	}

	if(!_ftpConnection)
	{
		_ftpStage				= SM_FTP_NOT_CONNECTED;
		_ftpStrLen				= 0;
		_ftpFlags.Val			= 0;
		_ftpCmd					= FTP_CMD_NONE;
		_ftpStageCmd			= SM_FTP_CMD_IDLE;
		_ftpTxSize				= 0;
		return;
	}

	if(_ftpTxSize)
	{
		_ftpTxSize = _SendData(_ftpConnectionSocket, _ftpTxBuf, _ftpTxSize);

		if(_ftpTxSize == INVALID_SOCKET)
		{
			TCPClose(_ftpConnectionSocket);
			return;
		}

		if(_ftpTxSize)
			return;
	}

	remoteInfo = TCPGetRemoteInfo(_ftpConnectionSocket);
	RemoteIP = remoteInfo->remote.IPAddr.Val;

	_ftpStrLen = MAX_FTP_CMD_STRING_LEN;
	
	if((result = _GetTCPRecv(_ftpConnectionSocket, _ftpStr, &_ftpStrLen)))
	{
		if(result == INVALID_SOCKET)
		{
			_CloseConnection(_ftpConnectionSocket);
			return;
		}

		_ftpStrLen--;

		if(_ftpStr[_ftpStrLen] == '\n')
		{
			_ftpStr[_ftpStrLen]		= 0;
			_ftpStrLen				= 0;
			_ftpNumArg				= _ParseFTPString(_ftpStr, _ftpArgv, MAX_FTP_ARGS);
			_ftpCmd					= _GetFTPCommand(_ftpArgv[0]);
		}
	}else
	{
		if(_ftpStage != SM_FTP_NOT_CONNECTED)
		{
			DWORD	curr_tick;

			curr_tick = TickGet();
			curr_tick = curr_tick - _lastActivity;

			if(curr_tick >= FTP_TIMEOUT)
			{
				_lastActivity	= TickGet();
				_ftpCmd			= FTP_CMD_QUIT;
				_ftpStage		= SM_FTP_CONNECTED; 
			}
		}
	}

	switch(_ftpStage)
	{
	case SM_FTP_NOT_CONNECTED:
		_ftpResponce	= FTP_RESP_BANNER;
		_lastActivity	= TickGet();

	case SM_FTP_RESPOND:
		strcpy((char *)_ftpTxBuf, _ftpResponseString[_ftpResponce]);
		_ftpTxSize = strlen((char *)_ftpTxBuf);
		_ftpTxSize = _SendData(_ftpConnectionSocket, _ftpTxBuf, _ftpTxSize);

		_ftpResponce	= FTP_RESP_NONE;
		_ftpStage		= SM_FTP_CONNECTED;

		if(_ftpTxSize)
			break;

	case SM_FTP_CONNECTED:
		if(_ftpCmd != FTP_CMD_NONE)
		{
			if(_ExecuteFTPCommand(_ftpCmd, _ftpArgv))
			{
				if(_ftpResponce != FTP_RESP_NONE)
					_ftpStage = SM_FTP_RESPOND;

				if(_ftpCmd == FTP_CMD_QUIT)
					_ftpStage = SM_FTP_NOT_CONNECTED;

				_ftpCmd			= FTP_CMD_NONE;
				_ftpStageCmd	= SM_FTP_CMD_IDLE;

			}else
			{
				if(_ftpResponce != FTP_RESP_NONE)
					_ftpStage = SM_FTP_RESPOND;
			}
		}
		break;

    default:
        break;
	}

	return;
}
/*********************************************************************
 * Function:		PUBLIC BOOL FTPGetUser(BYTE *user)      
 *
 * PreCondition:    None.
 *
 * Input:           user	- pointer to the user buffer
 *
 * Output:          true if there is a FTP connection, else false
 *
 * Side Effects:    None.
 *
 * Overview:        This function fills the user pointer with 
 *					FTP user information.
 ********************************************************************/
BOOL FTPGetUser(BYTE *user)
{
	if(!_ftpConnection)
	{
		strcpy((char *)user, "Not Connected");
		return FALSE;
	}

	if(_ftpUser[0] == 0)
	{
		strcpy((char *)user, "No user");
	}else
	{
		strcpy((char *)user, "User:");
		strcat((char *)user, (char *)_ftpUser);
	}

	return TRUE;
}
/*********************************************************************
 * Function:            PUBLIC FTP_COMMAND FTPGetCammand(void)      
 *
 * PreCondition:        None.   
 *
 * Input:               None.
 *
 * Output:              Current FTP command
 *
 * Side Effects:        None
 *
 * Overview:            This funciton will return the current 
 *                      FTP command
 *
 ********************************************************************/
SM_FTP FTPGetStage(void)
{
    return _ftpStage;
}   
/*********************************************************************
 * Function:        PRIVATE BOOL _AcceptConnection(void)
 *
 * PreCondition:    None.
 *
 * Input:           None.
 *
 * Output:          true if there is a FTP connection, else false
 *
 * Side Effects:    None.
 *
 * Overview:        This function accepts incoming connections on the 
 *					FTP port.  If there is an incoming connection, the 
 *					socket will be assigned and the service function will
 *					be able to Rx FTP commands.
 ********************************************************************/
static BOOL _AcceptConnection(void)
{
	if (TCPIsConnected(_ftpListenSocket))
	{
		_ftpConnectionSocket = _ftpListenSocket;
		_ftpUser[0] = 0;
		return TRUE;
	}
	else
			return FALSE;
}
/*********************************************************************
 * Function:        PRIVATE void _CloseConnection(SOCKET socket)
 *
 * PreCondition:    None.
 *
 * Input:           socket	- socket to close
 *
 * Output:          None.
 *
 * Side Effects:    None.
 *
 * Overview:        This function closes the socket.  If a file is 
 *					open, this function will close and initialize the
 *					pointer.  Other variables will be initialized as well
 ********************************************************************/
static void _CloseConnection(TCP_SOCKET socket)
{
	TCPClose (socket);

	if(_ftpFile)
		FSfclose(_ftpFile);

	_ftpFile				= NULL;
	_ftpStage				= SM_FTP_NOT_CONNECTED;
	_ftpStrLen				= 0;
	_ftpFlags.Val			= 0;
	_ftpCmd					= FTP_CMD_NONE;
	_ftpStageCmd			= SM_FTP_CMD_IDLE;
	_ftpConnection			= FALSE;
	_ftpTxSize				= 0;
	_ftpDataPort.Val		= FTP_DATA_PORT;

}
/*********************************************************************
 * Function:		PRIVATE int _GetTCPRecv(	SOCKET socket, 
 *												BYTE *data, 
 *												BYTE *data_len)        
 *
 * PreCondition:    None
 *
 * Input:           socket		- socket handle to get pending data from
 *					data		- pointer to the buffer to place the data
 *					data_len	- pointer to the size of the data gotten, it
 *									will also pass the max size of the data
 *
 * Output:          The result of the recv funciton
 *
 * Side Effects:    None
 *
 * Overview:        This function gets pending data from a socket.
 ********************************************************************/
static int _GetTCPRecv(TCP_SOCKET socket, BYTE *data, BYTE *data_len)
{
	int			result;

	result = 0;
	if (TCPIsGetReady(socket))
	{
		result = TCPGetArray (socket, data, *data_len);
		if(result > 0)
			*data_len = (BYTE)result;
	}
	return result;
	
}
/*********************************************************************
 * Function:        PRIVATE BYTE _ParseFTPString(	BYTE *ftp_str, 
 *													BYTE **arg, 
 *													BYTE max_arg)
 *
 * PreCondition:    None
 *
 * Input:           ftp_str	- pointer to an FTP command string
 *					arg		- pointer to a list of FTP command arguments
 *					max_arg - maximun number of arguments allowed
 *
 * Output:          number of agruments in the command string
 *
 * Side Effects:    None
 *
 * Overview:        This function parses the FTP command string into 
 *					a list of arguments.
 *
 * Note:			EXAMPLE:
 *					Rx FTP Cmd:	PORT,10,10,33,203,4,84
 *					Parsed Cmd:
 *						arg[0] = PORT
 *						arg[1] = 10
 *						arg[2] = 10
 *						arg[3] = 33
 *						arg[4] = 203
 *						arg[5] = 4
 *						arg[6] = 84
 ********************************************************************/
BYTE _ParseFTPString(BYTE *ftp_str, BYTE **arg, BYTE max_arg)
{
	BYTE	num;
	BYTE	alpha;
	BOOL	space;

	while(*ftp_str == ' ')
		ftp_str++;

	space	= FALSE;
	arg[0]	= ftp_str;
	num		= 1;

	while((alpha = *ftp_str) && (num < max_arg))
	{
		if(!space)
		{
			if(alpha == ' ' || alpha == ',')
			{
				*ftp_str	= 0;
				space		= TRUE;
			}else
			{
				if(alpha == '\r' || alpha == '\n')
					*ftp_str = 0;
			}
		
		}else
		{
			if(alpha != ' ')
			{
				arg[num++] = ftp_str;
				space = FALSE;
			}
		}

		ftp_str++;
	}

	return num;
}
/*********************************************************************
 * Function:        PRIVATE FTP_COMMAND _GetFTPCommand(BYTE *cmd_str)
 *
 * PreCondition:    None
 *
 * Input:           cmd_str - the parsed FTP command
 *
 * Output:          FTP_COMMAND enum, if none return FTP_CMD_UNKNOWN
 *					See ftp_private for the FTP_COMMAND enum
 *
 * Side Effects:    None.
 *
 * Overview:        This function takes the command string and returns 
 *					the enumeration that will be used to identify and 
 *					processes it.
 *
 * Note:			EXAMPLE
 *					Command String:     "USER"
 *					Return Value:		FTP_CMD_USER
 ********************************************************************/
FTP_COMMAND _GetFTPCommand(BYTE *cmd_str)
{
	FTP_COMMAND	cmd;

	for(cmd = FTP_CMD_USER; cmd < FTP_CMD_UNKNOWN; cmd++)
	{
		BYTE size;

		size = strlen(_ftpCommandString[cmd]);

		if(!memcmp(cmd_str, _ftpCommandString[cmd], size))
			break;
	}

	return cmd;
}
/*********************************************************************
 * Function:		PRIVATE BOOL _ExecuteFTPCommand(	FTP_COMMAND cmd, 
 *														BYTE **arg)
 *
 * PreCondition:    None.
 *
 * Input:           cmd - FTP_COMMAND (see ftp_private.h for FTP_COMMAND enum)
 *					arg - pointer to the list of arguments
 *
 * Output:          if command serviced true, else false
 *
 * Side Effects:    None.
 *
 * Overview:        This funciton services FTP commands and provides the 
 *					responces
 *
 * Note:			If the FTP command is unknown, the responce will be
 *					"502 Not implemented\r\n"
 ********************************************************************/
BOOL _ExecuteFTPCommand(FTP_COMMAND cmd, BYTE **arg)
{
	switch(cmd)
	{
	case FTP_CMD_USER:
		_ftpFlags.Bits.bUserSupplied	= TRUE;
		_ftpFlags.Bits.bLoggedIn		= FALSE;
		_ftpResponce					= FTP_RESP_USER_OK;
		memcpy(_ftpUser, arg[1], FTP_USER_NAME_LEN);
		break;

	case FTP_CMD_PASS:
		if(!_ftpFlags.Bits.bUserSupplied)
		{
			_ftpResponce = FTP_RESP_LOGIN;
		}else
		{
			if(FTPVerify(_ftpUser, arg[1]))
			{
				_ftpResponce				= FTP_RESP_PASS_OK;
				_ftpFlags.Bits.bLoggedIn	= TRUE;
			}else
			{
				_ftpResponce				= FTP_RESP_LOGIN;
			}
		}
		break;

	case FTP_CMD_PORT:
		_ftpDataPort.v[1]	= (BYTE)atoi((char *)arg[5]);
		_ftpDataPort.v[0]	= (BYTE)atoi((char *)arg[6]);
		_ftpResponce		= FTP_RESP_PORT;
		break;

	case FTP_CMD_ABORT:
		if(_ftpDataSocket != INVALID_SOCKET)
			TCPClose(_ftpDataSocket);

		if(_ftpFile)
			FSfclose(_ftpFile);
		
		_ftpFile = NULL;

		_ftpResponce	= FTP_RESP_OK;
		break;

	case FTP_CMD_QUIT:
		return _FTPQuit();
		
	case FTP_CMD_STOR:
		return _FTPPutFile(arg[1]);

	case FTP_CMD_RETR:
		return _FTPGetFile(arg[1]);

	case FTP_CMD_DELE:
		return _FTPDeleFile(arg[1]);

	case FTP_CMD_NLST:
		return _FTPListFile(arg);

	case FTP_CMD_LIST:
		return _FTPListFile(arg);

	case FTP_CMD_CWD:
		return _FTPCWD(arg[1]);

	case FTP_CMD_NOOP:
		_ftpResponce = FTP_RESP_OK;
		break;

	case FTP_CMD_PWD:
		_ftpResponce = FTP_RESP_DIRECTORY;
		break;

	case FTP_CMD_RMD:
		return _FTPDeleDirectory(arg[1]);

	case FTP_CMD_MKD:
		return _FTPMakeDirectory(arg[1]);

	case FTP_CMD_PASV:
		return _FTPPASV();

	case FTP_CMD_SYST:
	case FTP_CMD_TYPE:
		if (*arg[1] == 'A') _ftpResponce = FTP_RESP_ASCII;
		else 				_ftpResponce = FTP_RESP_SYST;
		break;

	default:
		_ftpResponce = FTP_RESP_UNKNOWN;
		break;

	}

	return TRUE;
}
/*********************************************************************
 * Function:        PRIVATE BOOL _FTPQuit(void)
 *
 * PreCondition:	None   
 *
 * Input:           None
 *
 * Output:          Always false.
 *
 * Side Effects:    None
 *
 * Overview:        This function properly closes all open sockets
 *					and files.  It will also re-initialize all parameters
 ********************************************************************/
BOOL _FTPQuit(void)
{
	switch(_ftpStageCmd)
	{
	default:
	case SM_FTP_CMD_IDLE:
	case SM_FTP_CMD_WAIT:
	case SM_FTP_CMD_RX_TX:
		if(_ftpDataSocket != INVALID_SOCKET)
			TCPClose(_ftpDataSocket);

		if(_ftpFile)
			FSfclose(_ftpFile);

		_ftpResponce			= FTP_RESP_QUIT_OK;
		_ftpDataSocket			= INVALID_SOCKET;
		_ftpFile				= NULL;
		_ftpStageCmd			= SM_FTP_CMD_WAIT_FOR_DISCONNECT;
		_ftpPassive 			= FALSE;
		break;

	case SM_FTP_CMD_WAIT_FOR_DISCONNECT:
		if(_ftpConnectionSocket != INVALID_SOCKET) 
			TCPClose(_ftpConnectionSocket);
		_ftpListenSocket = INVALID_SOCKET;
		_ftpPassive 			= FALSE;
		break;
	}

	return FALSE;
}
/*********************************************************************
 * Function:		PRIVATE BOOL _FTPPutFile(BYTE *filename)     
 *
 * PreCondition:    None.
 *
 * Input:           filename - pointer to the ASCII file name to 
 *								put on the server
 *
 * Output:          true if competed, else false
 *
 * Side Effects:    None.
 *
 * Overview:        This function will open a socket on the user supplied
 *					port and connect to it.  It will then create a file to store
 *					all of the information coming accross the "data" port.
 ********************************************************************/
BOOL _FTPPutFile(BYTE *filename)
{
	static	int result;

	switch(_ftpStageCmd)
	{
	case SM_FTP_CMD_IDLE:
		if(!_ftpFlags.Bits.bLoggedIn)
		{
			_ftpResponce	= FTP_RESP_LOGIN;
			return TRUE;
		}else
		{
			if (_ftpPassive)
			{
				_ftpDataSocket = TCPOpen (0, TCP_OPEN_SERVER, PassivePort, TCP_PURPOSE_FTP_DATA); 
			} else {
				_ftpDataSocket = TCPOpen (RemoteIP, TCP_OPEN_IP_ADDRESS, _ftpDataPort.Val, TCP_PURPOSE_FTP_DATA);
			}
			_ftpResponce	= FTP_RESP_DATA_OPEN;
			_ftpStageCmd = SM_FTP_CMD_WAIT;

			if(_ftpFile)
				FSfclose(_ftpFile);

			_ftpFile = FSfopen((char *)filename, FS_WRITE);

			if(!_ftpFile)
			{
				_ftpResponce = FTP_RESP_UNKNOWN;
				TCPClose(_ftpDataSocket);
				_ftpDataSocket = INVALID_SOCKET;
				return TRUE;
			}
		}
		break;


	case SM_FTP_CMD_WAIT:
		if (TCPIsConnected(_ftpDataSocket))
				_ftpStageCmd = SM_FTP_CMD_RX_TX;
		break;

	case SM_FTP_CMD_RX_TX:
		size = 64;
		result = _GetTCPRecv(_ftpDataSocket, buf, &size);
		if (!result)
		{
			if (!TCPIsConnected(_ftpDataSocket))
			{
				TCPClose(_ftpDataSocket);
				FSfclose(_ftpFile);

				_ftpDataSocket	= INVALID_SOCKET;
				_ftpFile		= NULL;
				_ftpResponce	= FTP_RESP_DATA_CLOSE;
				return TRUE;
			}
		}
		else
		{ 
			FSfwrite(buf, 1, result, _ftpFile);
		}
		return FALSE;
			
	default:
		break;
	}
	return FALSE;
}

/*********************************************************************
 * Function:		PRIVATE BOOL _FTPGetFile(BYTE *filename)     
 *
 * PreCondition:    None.
 *
 * Input:           filename - pointer to the ASCII file name to 
 *								get for the server
 *
 * Output:          true if competed, else false
 *
 * Side Effects:    None.
 *
 * Overview:        This function will open a socket on the user supplied
 *					port and connect to it.  It will then open the file and 
 *					pass the information in it to the client.
 ********************************************************************/
BOOL _FTPGetFile(BYTE *filename)
{
	static	BYTE	filename_temp[13];
	static	BYTE	name_len, i;
	static	int result;

	switch(_ftpStageCmd)
	{
	case SM_FTP_CMD_IDLE:
		if(!_ftpFlags.Bits.bLoggedIn)
		{
			_ftpResponce	= FTP_RESP_LOGIN;
			return TRUE;
		}else
		{
			if (_ftpPassive)
			{
				_ftpDataSocket = TCPOpen (0, TCP_OPEN_SERVER, PassivePort, TCP_PURPOSE_FTP_DATA); 
				_ftpStageCmd = SM_FTP_CMD_WAIT;
			} else {
				_ftpDataSocket = TCPOpen (RemoteIP, TCP_OPEN_IP_ADDRESS, _ftpDataPort.Val, TCP_PURPOSE_FTP_DATA);
				_ftpStageCmd = SM_FTP_CMD_RX_TX;
			}

			_ftpResponce = FTP_RESP_DATA_OPEN;

			if(_ftpFile) FSfclose(_ftpFile);
			_ftpFile		= FSfopen((char *)filename, FS_BINARYREAD);

			if(!_ftpFile)
			{
				memset (filename_temp,0x00, 13);
				name_len = strlen((char *)filename);
				if (name_len > 12) name_len = 12;
				for (i = 0 ; i < name_len ; i++)
				{
					filename_temp[i] = toupper (filename[i]);
				}	
				_ftpFile		= FSfopen((char *)filename_temp, FS_BINARYREAD);
				if(!_ftpFile)
				{
					_ftpResponce = FTP_RESP_UNKNOWN;
					TCPClose(_ftpDataSocket);
					_ftpDataSocket = INVALID_SOCKET;
					return TRUE;
				}
			}
		}
		break;


	case SM_FTP_CMD_WAIT:
		if (!TCPIsConnected(_ftpDataSocket)) break;
		_ftpStageCmd = SM_FTP_CMD_RX_TX;
		break;

	case SM_FTP_CMD_RX_TX:
			result = FSfread(buf, 64, 1, _ftpFile);
			if(!result)
			{
				TCPClose(_ftpDataSocket);
				FSfclose(_ftpFile);

				_ftpDataSocket	= INVALID_SOCKET;
				_ftpFile		= NULL;
				_ftpResponce	= FTP_RESP_DATA_CLOSE;
				return TRUE;
			}
			
			if (TCPIsPutReady(_ftpDataSocket) < result)
			{
				_ftpStageCmd = SM_FTP_CMD_RX_TX_RETRY;
				break;
			}
			TCPPutArray (_ftpDataSocket, buf, result);
		break;

	case SM_FTP_CMD_RX_TX_RETRY:
			if (TCPIsPutReady(_ftpDataSocket) < result) break;
			TCPPutArray (_ftpDataSocket, buf, result);
			_ftpStageCmd = SM_FTP_CMD_RX_TX;
		break;

	default:
		break;
	}
	return FALSE;

}
/*********************************************************************
 * Function:		PRIVATE int _SendData(	SOCKET socket, 
 *											BYTE *buf, 
 *											int len)
 *
 * PreCondition:	None    
 *
 * Input:           socket	- socket handle 
 *					buf		- buffer of data to send
 *					len		- size of data in bytes to transfer
 *
 * Output:          result of the send function
 *
 * Side Effects:    none
 *
 * Overview:        This funciton sends data on the TCP/IP socket. 
 *					If the data is not sent, it will return the remaining
 *					amount that needs to be sent.
 ********************************************************************/
int _SendData(TCP_SOCKET socket, BYTE *buf, int len)
{
    int result;
	int i, size;


	result = TCPIsPutReady(socket);
	if (result >= len ) result = len;
	if (result)
	{
		TCPPutArray (socket, buf, result);
		if (result == len) return 0;
		
		size = len - result;			// bytes left in the buffer
	
		if(size)
		{
			for(i = 0; i < size; i++)		// move the buffer 
				buf[i] = buf[result + i];	
		}
		return size;
	} else {
		return len;
	}
}

BOOL _FTPDeleFile(BYTE *filename)
{
	static	BYTE	filename_temp[13];
	static	BYTE	name_len, i;

	switch(_ftpStageCmd)
	{
		case SM_FTP_CMD_IDLE:
			if(!_ftpFlags.Bits.bLoggedIn)
			{
				_ftpResponce	= FTP_RESP_LOGIN;
				return TRUE;
			}else
			{
				if(_ftpFile) FSfclose(_ftpFile);

				if (FSremove ((char *)filename))
				{
					memset (filename_temp,0x00, 13);
					name_len = strlen((char *)filename);
					if (name_len > 12) name_len = 12;
					for (i = 0 ; i < name_len ; i++)
					{
						filename_temp[i] = toupper (filename[i]);
					}
					FSremove ((char *)filename_temp);
				}
				_ftpResponce	= FTP_RESP_DELETE;
				return TRUE;
			}
			break;

		default:
			break;
	}
	return FALSE;

}

BOOL _FTPDeleDirectory(BYTE *directory)
{
	static	BYTE	directory_temp[13];
	static	BYTE	directory_len, i;

	switch(_ftpStageCmd)
	{
		case SM_FTP_CMD_IDLE:
			if(!_ftpFlags.Bits.bLoggedIn)
			{
				_ftpResponce	= FTP_RESP_LOGIN;
				return TRUE;
			}else
			{
				if (FSrmdir ((char *)directory, FALSE))
				{
					memset (directory_temp,0x00, 13);
					directory_len = strlen((char *)directory);
					if (directory_len > 12) directory_len = 12;
					for (i = 0 ; i < directory_len ; i++)
					{
						directory_temp[i] = toupper (directory[i]);
					}
					FSrmdir ((char *)directory_temp, FALSE);
				}
				_ftpResponce	= FTP_RESP_DELETE;
				return TRUE;
			}
			break;

		default:
			break;
	}
	return FALSE;

}

BOOL _FTPMakeDirectory(BYTE *directory)
{
	switch(_ftpStageCmd)
	{
		case SM_FTP_CMD_IDLE:
			if(!_ftpFlags.Bits.bLoggedIn)
			{
				_ftpResponce	= FTP_RESP_LOGIN;
				return TRUE;
			}else
			{
				FSmkdir ((char *)directory);
				_ftpResponce	= FTP_RESP_DELETE;
				return TRUE;
			}
			break;

		default:
			break;
	}
	return FALSE;

}


static	BOOL _FTPListFile(BYTE *arg[])
{
	static	WORD	ret;
	static	SearchRec recFile;
	static	BYTE	attr = ATTR_READ_ONLY | ATTR_ARCHIVE | ATTR_DIRECTORY;
	static	BYTE	temp[20], i;
	static	WORD	year;
	static	BYTE	month, day, hours, minutes, seconds;
	static	BYTE	k;

	switch(_ftpStageCmd)
	{
		case SM_FTP_CMD_IDLE:
			if(!_ftpFlags.Bits.bLoggedIn)
			{
				_ftpResponce	= FTP_RESP_LOGIN;
				return TRUE;
			}
			else
			{
				if (_ftpPassive)
				{
					_ftpDataSocket = TCPOpen (0, TCP_OPEN_SERVER, PassivePort, TCP_PURPOSE_FTP_DATA); 
					_ftpStageCmd = SM_FTP_CMD_WAIT;
				} else {
					_ftpDataSocket = TCPOpen (RemoteIP, TCP_OPEN_IP_ADDRESS, _ftpDataPort.Val, TCP_PURPOSE_FTP_DATA);
					_ftpStageCmd = SM_FTP_CMD_RX_TX;
				}
				_ftpResponce = FTP_RESP_LIST;
			}

			memcpy (temp, arg[1], 2);
			k = 1;
			if (!memcmp (temp, "-a" ,2)) k = 2;

			if ((k == 1) && strlen((char *)arg[k]))
			{
		    	ret = FindFirst  ((char *)arg[k], attr, &recFile);
			} else {
		    	ret = FindFirst  ((char *)&"*.*", attr, &recFile);
			}
			if (ret)
			{
				TCPClose(_ftpDataSocket);
				_ftpDataSocket	= INVALID_SOCKET;
				_ftpResponce	= FTP_RESP_LIST_DONE;
				return TRUE;
			}
			break;

		case SM_FTP_CMD_WAIT:
			if (!TCPIsConnected(_ftpDataSocket)) break;
			_ftpStageCmd = SM_FTP_CMD_RX_TX;
			break;

		case SM_FTP_CMD_RX_TX:
			if (recFile.utf16LFNfoundLength)
			{
				if (TCPIsPutReady(_ftpDataSocket) < (recFile.utf16LFNfoundLength + 32)) break;
			} else {
				if (TCPIsPutReady(_ftpDataSocket) < 60) break;
			}
			seconds = recFile.timestamp & 0x00000001f * 2;
			minutes =  (recFile.timestamp >> 5) & 0x00000003f;
			hours =  (recFile.timestamp >> 11) & 0x00000001f;
			day =  (recFile.timestamp >> 16) & 0x00000001f;
			month =  (recFile.timestamp >> 21) & 0x00000000f;
			year =  (recFile.timestamp >> 25);
			year += 1980;
			sprintf ((char *)temp, "%04d/%02d/%02d  %02d:%02d  ", year, month, day, hours, minutes);
			TCPPutArray (_ftpDataSocket, temp, 18);
		
			if (recFile.attributes & ATTR_DIRECTORY)
			{
				TCPPutArray (_ftpDataSocket, (BYTE *)&" <DIR>      ", 12);
			} else {
				sprintf ((char *)temp,"%11ld ",recFile.filesize);
				TCPPutArray (_ftpDataSocket, (BYTE *)&temp, 12);
			}
			if (recFile.utf16LFNfoundLength)
			{
				for (i = 0 ; i < recFile.utf16LFNfoundLength ; i++)
				{
					TCPPut(_ftpDataSocket, recordFoundName[i] & 0xff);
				}
			} else {
				TCPPutArray (_ftpDataSocket, (BYTE *)&recFile.filename, strlen(recFile.filename));
			}
			TCPPutArray (_ftpDataSocket, (BYTE *)&"\r\n", 2);
			ret = FindNext (&recFile);
			if (ret)
			{
				TCPClose(_ftpDataSocket);
				_ftpDataSocket	= INVALID_SOCKET;
				_ftpResponce	= FTP_RESP_LIST_DONE;
				return TRUE;
			}
			break;

		default:
			break;
	}
	return FALSE;

}

BOOL _FTPCWD(BYTE *directory)
{
	switch(_ftpStageCmd)
	{
		case SM_FTP_CMD_IDLE:
			if(!_ftpFlags.Bits.bLoggedIn)
			{
				_ftpResponce	= FTP_RESP_LOGIN;
				return TRUE;
			}else
			{
				FSchdir ((char *)directory);
				_ftpResponce	= FTP_RESP_CWD;
				return TRUE;
			}
			break;

		default:
			break;
	}
	return FALSE;

}

BOOL _FTPPASV()
{
	static BYTE temp[30];

	switch(_ftpStageCmd)
	{
		case SM_FTP_CMD_IDLE:
			if(!_ftpFlags.Bits.bLoggedIn)
			{
				_ftpResponce	= FTP_RESP_LOGIN;
				return TRUE;
			}else
			{
				_ftpPassive = TRUE;
				sprintf ((char *)temp,"(%-d,%-d,%-d,%-d,%-d,%-d).",
					AppConfig.MyIPAddr.byte.LB, AppConfig.MyIPAddr.byte.HB, AppConfig.MyIPAddr.byte.UB, AppConfig.MyIPAddr.byte.MB,
					PassivePort/256, PassivePort%256);
//				_ftpDataSocket = TCPOpen (0, TCP_OPEN_SERVER, 20, TCP_PURPOSE_FTP_DATA); 
				TCPPutArray (_ftpConnectionSocket, (BYTE *)&"227 Entering Passive Mode ", 26);
				TCPPutArray (_ftpConnectionSocket, (BYTE *)&temp, strlen((char *)temp));
				TCPPutArray (_ftpConnectionSocket, (BYTE *)&"\r\n", 2);
				_ftpResponce	= FTP_RESP_CWD;
				return TRUE;
			}
			break;

		default:
			break;
	}
	return FALSE;

}


BOOL FTPVerify(BYTE *user, BYTE *password)
{
	// TODO: ADD Code to look at the user and password

	if (!strcmp((char *)AppConfig.UserID, (char *)user) && !strcmp((char *)AppConfig.PASSWORD, (char *)password))
	{
		_ftpPassive = FALSE;
		return TRUE;
	}
	return FALSE;
}
